<?php

declare(strict_types=1);

use PHPUnit\Framework\TestCase;
use hu\jmk\rtwi\Request;

final class RequestTest extends TestCase {
  /**
   * @test
   */
  public function shouldGetDefaultAjaxForMissingAjax():void {
    $this->assertFalse((new Request(array()))->isAjax());
  }

  /**
   * @test
   */
  public function shouldGetDefaultAjaxForInvalidAjax():void {
    $this->assertFalse((new Request(array("ajax" => "")))->isAjax());
    $this->assertFalse((new Request(array("ajax" => "0")))->isAjax());
  }

  /**
   * @test
   */
  public function shouldGetAjax():void {
    $this->assertTrue((new Request(array("ajax" => "1")))->isAjax());
    $this->assertTrue((new Request(array("ajax" => "meh")))->isAjax());
  }

  /**
   * @test
   */
  public function shouldGetDefaultFormatForMissingFormat():void {
    $this->assertSame(Request::FORMAT_HTML, (new Request(array()))->getFormat());
  }

  /**
   * @test
   */
  public function shouldGetDefaultFormatForInvalidFormat():void {
    $this->assertSame(Request::FORMAT_HTML, (new Request(array("format" => "meh")))->getFormat());
  }

  /**
   * @test
   */
  public function shouldGetFormat():void {
    $this->assertCount(2, Request::FORMATS);
    $this->assertSame(Request::FORMAT_HTML, (new Request(array("format" => "html")))->getFormat());
    $this->assertSame(Request::FORMAT_XML, (new Request(array("format" => "xml")))->getFormat());
  }

  /**
   * @test
   */
  public function shouldNotHaveHashForMissingHash():void {
    $this->assertFalse((new Request(array()))->hasHash());
  }

  /**
   * @test
   */
  public function shouldNotGetHashForMissingHash():void {
    $this->expectException(TypeError::class);
    (new Request(array()))->getHash();
  }

  /**
   * @test
   */
  public function shouldGetHash():void {
    $this->assertSame("0123456789abcdef", (new Request(array("hash" => "0123456789abcdef")))->getHash());
  }

  /**
   * @test
   */
  public function shouldNotHaveIdForMissingId():void {
    $this->assertFalse((new Request(array()))->hasId());
  }

  /**
   * @test
   */
  public function shouldGetDefaultIdForMissingId():void {
    $this->assertSame(-1, (new Request(array()))->getId());
  }

  /**
   * @test
   */
  public function shouldNotHaveIdForInvalidId():void {
    $this->assertFalse((new Request(array("id" => "")))->hasId());
    $this->assertFalse((new Request(array("id" => "meh")))->hasId());
  }

  /**
   * @test
   */
  public function shouldGetDefaultIdForInvalidId():void {
    $this->assertSame(-1, (new Request(array("id" => "")))->getId());
    $this->assertSame(-1, (new Request(array("id" => "meh")))->getId());
  }

  /**
   * @test
   */
  public function shouldGetId():void {
    $this->assertSame(1, (new Request(array("id" => "1")))->getId());
  }

  /**
   * @test
   */
  public function shouldGetDefaultModForMissingMod():void {
    $this->assertSame(Request::MOD_TORRENTS, (new Request(array()))->getMod());
  }

  /**
   * @test
   */
  public function shouldGetDefaultModForInvalidMod():void {
    $this->assertSame(Request::MOD_TORRENTS, (new Request(array("mod" => "meh")))->getMod());
  }

  /**
   * @test
   */
  public function shouldGetMod():void {
    $this->assertCount(6, Request::MODS);
    $this->assertSame(Request::MOD_ADDTORRENT, (new Request(array("mod" => "addtorrent")))->getMod());
    $this->assertSame(Request::MOD_GETFILE, (new Request(array("mod" => "getfile")))->getMod());
    $this->assertSame(Request::MOD_LOGIN, (new Request(array("mod" => "login")))->getMod());
    $this->assertSame(Request::MOD_SERVERINFO, (new Request(array("mod" => "serverinfo")))->getMod());
    $this->assertSame(Request::MOD_TORRENT, (new Request(array("mod" => "torrent")))->getMod());
    $this->assertSame(Request::MOD_TORRENTS, (new Request(array("mod" => "torrents")))->getMod());
  }

  /**
   * @test
   */
  public function shouldNotAllowNullUpdateForMod():void {
    $request = new Request(array("mod" => "addtorrent"));
    $this->assertSame(Request::MOD_ADDTORRENT, $request->getMod());

    $request->setMod(null);
    $this->assertSame(Request::MOD_TORRENTS, $request->getMod());
  }

  /**
   * @test
   */
  public function shouldGetDefaultModAfterInvalidUpdate():void {
    $request = new Request(array("mod" => "addtorrent"));
    $this->assertSame(Request::MOD_ADDTORRENT, $request->getMod());

    $request->setMod("meh");
    $this->assertSame(Request::MOD_TORRENTS, $request->getMod());
  }

  /**
   * @test
   */
  public function shouldGetUpdatedMod():void {
    $request = new Request(array("mod" => "addtorrent"));
    $this->assertSame(Request::MOD_ADDTORRENT, $request->getMod());

    $request->setMod(Request::MOD_GETFILE);
    $this->assertSame(Request::MOD_GETFILE, $request->getMod());
  }

  /**
   * @test
   */
  public function shouldGetDefaultPageForMissingPage():void {
    $this->assertSame(Request::PAGE_INFO, (new Request(array()))->getPage());
  }

  /**
   * @test
   */
  public function shouldGetDefaultPageForInvalidPage():void {
    $this->assertSame(Request::PAGE_INFO, (new Request(array("page" => "meh")))->getPage());
  }

  /**
   * @test
   */
  public function shouldGetPage():void {
    $this->assertCount(6, Request::PAGES);
    $this->assertSame(Request::PAGE_CHUNKS, (new Request(array("page" => "chunks")))->getPage());
    $this->assertSame(Request::PAGE_FILES, (new Request(array("page" => "files")))->getPage());
    $this->assertSame(Request::PAGE_INFO, (new Request(array("page" => "info")))->getPage());
    $this->assertSame(Request::PAGE_PEERS, (new Request(array("page" => "peers")))->getPage());
    $this->assertSame(Request::PAGE_TRACKERS, (new Request(array("page" => "trackers")))->getPage());
    $this->assertSame(Request::PAGE_TRANSFERS, (new Request(array("page" => "transfers")))->getPage());
  }

  /**
   * @test
   */
  public function shouldNotHavePathForMissingPath():void {
    $this->assertFalse((new Request(array()))->hasPath());
  }

  /**
   * @test
   */
  public function shouldGetDefaultPathForMissingPath():void {
    $this->expectException(TypeError::class);
    (new Request(array()))->getPath();
  }

  /**
   * @test
   */
  public function shouldGetPath():void {
    $this->assertSame("some/path", (new Request(array("path" => "some/path")))->getPath());
  }
}
